<?php
/**
 * SupplierCheckController - Tedarikci Cek Takibi
 */
class SupplierCheckController {
    private $db;
    private $auth;

    public function __construct() {
        $this->db = Database::getInstance();
        $this->auth = new Auth();
    }

    public function index() {
        $this->ensureChecksTable();

        $checks = $this->db->select("
            SELECT
                sc.id,
                sc.supplier_id,
                sc.supplier_transaction_id,
                sc.check_number,
                sc.bank_name,
                sc.amount,
                sc.issue_date,
                sc.due_date,
                sc.status,
                sc.description,
                sc.created_at,
                s.supplier_name,
                st.reference_number
            FROM supplier_checks sc
            INNER JOIN suppliers s ON s.id = sc.supplier_id
            INNER JOIN supplier_transactions st ON st.id = sc.supplier_transaction_id
            ORDER BY sc.due_date ASC, sc.created_at DESC
        ");

        $summary = $this->db->selectOne("
            SELECT
                COUNT(*) as total_count,
                SUM(CASE WHEN status = 'BEKLEMEDE' THEN amount ELSE 0 END) as pending_total,
                SUM(CASE WHEN status = 'TAHSIL_EDILDI' THEN amount ELSE 0 END) as collected_total
            FROM supplier_checks
        ");

        $auth = $this->auth;
        $pageTitle = 'Cek Takibi';
        $currentPage = 'supplier-checks';

        require_once __DIR__ . '/../views/supplier-checks/index.php';
    }

    public function markCollected($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/supplier-checks');
        }

        $this->ensureChecksTable();

        $check = $this->db->selectOne("SELECT * FROM supplier_checks WHERE id = ?", [$id]);
        if (!$check) {
            Helper::setFlash('danger', 'Cek kaydi bulunamadi.');
            Helper::redirect('/supplier-checks');
        }

        $updated = $this->db->update('supplier_checks', [
            'status' => 'TAHSIL_EDILDI'
        ], 'id = ?', [$id]);

        if ($updated !== false) {
            Helper::setFlash('success', 'Cek tahsil edildi olarak isaretlendi.');
        } else {
            Helper::setFlash('danger', 'Islem basarisiz oldu.');
        }

        Helper::redirect('/supplier-checks');
    }

    public function markPending($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/supplier-checks');
        }

        $this->ensureChecksTable();

        $check = $this->db->selectOne("SELECT * FROM supplier_checks WHERE id = ?", [$id]);
        if (!$check) {
            Helper::setFlash('danger', 'Cek kaydi bulunamadi.');
            Helper::redirect('/supplier-checks');
        }

        $updated = $this->db->update('supplier_checks', [
            'status' => 'BEKLEMEDE'
        ], 'id = ?', [$id]);

        if ($updated !== false) {
            Helper::setFlash('success', 'Cek beklemede olarak guncellendi.');
        } else {
            Helper::setFlash('danger', 'Islem basarisiz oldu.');
        }

        Helper::redirect('/supplier-checks');
    }

    private function ensureChecksTable() {
        $this->db->query("
            CREATE TABLE IF NOT EXISTS supplier_checks (
                id INT PRIMARY KEY AUTO_INCREMENT,
                supplier_id INT NOT NULL,
                supplier_transaction_id INT NOT NULL,
                check_number VARCHAR(100) NOT NULL,
                bank_name VARCHAR(100),
                amount DECIMAL(15,2) NOT NULL,
                issue_date DATE NOT NULL,
                due_date DATE NOT NULL,
                status ENUM('BEKLEMEDE', 'TAHSIL_EDILDI', 'IADE') DEFAULT 'BEKLEMEDE',
                description TEXT,
                created_by INT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                UNIQUE KEY unique_supplier_transaction (supplier_transaction_id),
                INDEX idx_due_date (due_date),
                INDEX idx_status (status),
                INDEX idx_supplier (supplier_id),
                FOREIGN KEY (supplier_id) REFERENCES suppliers(id) ON DELETE CASCADE,
                FOREIGN KEY (supplier_transaction_id) REFERENCES supplier_transactions(id) ON DELETE CASCADE,
                FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_turkish_ci
        ");
    }
}
