<?php
/**
 * CheckImportController - Çek Verilerini İçe Aktarma
 */
class CheckImportController {
    private $db;
    private $auth;

    public function __construct() {
        $this->db = Database::getInstance();
        $this->auth = new Auth();
    }

    /**
     * Import both issued and received checks from Excel data
     */
    public function importChecks() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/import');
        }

        $this->ensureTablesExist();

        // Get check data from POST
        $verilenCekler = json_decode($_POST['verilen_cekler'] ?? '[]', true);
        $alinanCekler = json_decode($_POST['alinan_cekler'] ?? '[]', true);

        $stats = [
            'verilen_success' => 0,
            'verilen_failed' => 0,
            'alinan_success' => 0,
            'alinan_failed' => 0,
            'suppliers_created' => 0,
            'errors' => []
        ];

        try {
            $this->db->beginTransaction();

            // Process Verilen Çekler (Issued Checks to Suppliers)
            foreach ($verilenCekler as $cek) {
                try {
                    $this->processVerilenCek($cek, $stats);
                } catch (Exception $e) {
                    $stats['verilen_failed']++;
                    $stats['errors'][] = "Verilen Çek Hatası: " . $e->getMessage();
                }
            }

            // Process Alınan Çekler (Received Checks)
            foreach ($alinanCekler as $cek) {
                try {
                    $this->processAlinanCek($cek, $stats);
                } catch (Exception $e) {
                    $stats['alinan_failed']++;
                    $stats['errors'][] = "Alınan Çek Hatası: " . $e->getMessage();
                }
            }

            $this->db->commit();

            $message = sprintf(
                'Çekler başarıyla içe aktarıldı. Verilen: %d başarılı, %d başarısız. Alınan: %d başarılı, %d başarısız. %d yeni tedarikçi oluşturuldu.',
                $stats['verilen_success'],
                $stats['verilen_failed'],
                $stats['alinan_success'],
                $stats['alinan_failed'],
                $stats['suppliers_created']
            );

            Helper::setFlash('success', $message);

            if (!empty($stats['errors'])) {
                Helper::setFlash('warning', 'Bazı hatalar: ' . implode(', ', array_slice($stats['errors'], 0, 5)));
            }

        } catch (Exception $e) {
            $this->db->rollback();
            Helper::setFlash('danger', 'Çekler içe aktarılırken hata oluştu: ' . $e->getMessage());
        }

        Helper::redirect('/import');
    }

    /**
     * Process a single issued check (Verilen Çek)
     */
    private function processVerilenCek($cekData, &$stats) {
        // Required fields: tedarikci_adi, banka_adi, tutar, vade_tarihi
        $tedarikcıAdi = trim($cekData['tedarikci_adi'] ?? '');
        $bankaAdi = trim($cekData['banka_adi'] ?? '');
        $tutar = floatval($cekData['tutar'] ?? 0);
        $vadeTarihi = $cekData['vade_tarihi'] ?? '';
        $aciklama = $cekData['aciklama'] ?? null;

        if (empty($tedarikcıAdi) || empty($tutar) || empty($vadeTarihi)) {
            throw new Exception("Verilen çek için zorunlu alanlar eksik: " . $tedarikcıAdi);
        }

        // Find or create supplier
        $supplier = $this->db->selectOne(
            "SELECT id FROM suppliers WHERE supplier_name = ? LIMIT 1",
            [$tedarikcıAdi]
        );

        if (!$supplier) {
            // Create new supplier
            $supplierId = $this->db->insert('suppliers', [
                'supplier_name' => $tedarikcıAdi,
                'amount' => 0,
                'status' => 'pending',
                'description' => 'Çek içe aktarımı sırasında otomatik oluşturuldu',
                'created_by' => $this->auth->userId()
            ]);
            $stats['suppliers_created']++;
        } else {
            $supplierId = $supplier['id'];
        }

        // Create supplier transaction for payment
        $transactionId = $this->db->insert('supplier_transactions', [
            'supplier_id' => $supplierId,
            'transaction_type' => 'payment',
            'amount' => $tutar,
            'transaction_date' => $vadeTarihi,
            'reference_number' => $cekData['cek_no'] ?? null,
            'description' => $aciklama ? $aciklama . ' | Ödeme Yöntemi: Çek' : 'Ödeme Yöntemi: Çek',
            'created_by' => $this->auth->userId()
        ]);

        if (!$transactionId) {
            throw new Exception("İşlem kaydı oluşturulamadı");
        }

        // Create check record
        $checkId = $this->db->insert('supplier_checks', [
            'supplier_id' => $supplierId,
            'supplier_transaction_id' => $transactionId,
            'check_number' => $cekData['cek_no'] ?? 'Belirsiz',
            'bank_name' => $bankaAdi ?: null,
            'amount' => $tutar,
            'issue_date' => $vadeTarihi,
            'due_date' => $vadeTarihi,
            'status' => 'BEKLEMEDE',
            'description' => $aciklama,
            'created_by' => $this->auth->userId()
        ]);

        if (!$checkId) {
            throw new Exception("Çek kaydı oluşturulamadı");
        }

        $stats['verilen_success']++;
    }

    /**
     * Process a single received check (Alınan Çek)
     */
    private function processAlinanCek($cekData, &$stats) {
        // Required fields: musteri_adi, tutar, vade_tarihi
        $musteriAdi = trim($cekData['musteri_adi'] ?? '');
        $bankaAdi = trim($cekData['banka_adi'] ?? '');
        $tutar = floatval($cekData['tutar'] ?? 0);
        $vadeTarihi = $cekData['vade_tarihi'] ?? '';
        $aciklama = $cekData['aciklama'] ?? null;

        if (empty($musteriAdi) || empty($tutar) || empty($vadeTarihi)) {
            throw new Exception("Alınan çek için zorunlu alanlar eksik: " . $musteriAdi);
        }

        // Create received check record (no customer tracking yet as mentioned in requirements)
        $checkId = $this->db->insert('received_checks', [
            'customer_name' => $musteriAdi,
            'bank_name' => $bankaAdi ?: null,
            'amount' => $tutar,
            'due_date' => $vadeTarihi,
            'check_number' => $cekData['cek_no'] ?? null,
            'status' => 'BEKLEMEDE',
            'description' => $aciklama,
            'created_by' => $this->auth->userId()
        ]);

        if (!$checkId) {
            throw new Exception("Alınan çek kaydı oluşturulamadı");
        }

        $stats['alinan_success']++;
    }

    /**
     * Ensure all necessary tables exist
     */
    private function ensureTablesExist() {
        // Ensure supplier_checks table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS supplier_checks (
                id INT PRIMARY KEY AUTO_INCREMENT,
                supplier_id INT NOT NULL,
                supplier_transaction_id INT NOT NULL,
                check_number VARCHAR(100) NOT NULL,
                bank_name VARCHAR(100),
                amount DECIMAL(15,2) NOT NULL,
                issue_date DATE NOT NULL,
                due_date DATE NOT NULL,
                status ENUM('BEKLEMEDE', 'TAHSIL_EDILDI', 'IADE') DEFAULT 'BEKLEMEDE',
                description TEXT,
                created_by INT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                UNIQUE KEY unique_supplier_transaction (supplier_transaction_id),
                INDEX idx_due_date (due_date),
                INDEX idx_status (status),
                INDEX idx_supplier (supplier_id),
                FOREIGN KEY (supplier_id) REFERENCES suppliers(id) ON DELETE CASCADE,
                FOREIGN KEY (supplier_transaction_id) REFERENCES supplier_transactions(id) ON DELETE CASCADE,
                FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_turkish_ci
        ");

        // Ensure supplier_transactions table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS supplier_transactions (
                id INT PRIMARY KEY AUTO_INCREMENT,
                supplier_id INT NOT NULL,
                transaction_type ENUM('purchase', 'payment', 'refund', 'adjustment') NOT NULL,
                amount DECIMAL(15,2) NOT NULL,
                transaction_date DATE NOT NULL,
                reference_number VARCHAR(100),
                description TEXT,
                created_by INT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                FOREIGN KEY (supplier_id) REFERENCES suppliers(id) ON DELETE CASCADE,
                FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL,
                INDEX idx_supplier_trans (supplier_id, transaction_date),
                INDEX idx_trans_type (transaction_type),
                INDEX idx_trans_date (transaction_date)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_turkish_ci
        ");

        // Ensure received_checks table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS received_checks (
                id INT PRIMARY KEY AUTO_INCREMENT,
                customer_name VARCHAR(200) NOT NULL,
                bank_name VARCHAR(100),
                amount DECIMAL(15,2) NOT NULL,
                due_date DATE NOT NULL,
                check_number VARCHAR(100),
                status ENUM('BEKLEMEDE', 'TAHSIL_EDILDI', 'IADE', 'CIRO_EDILDI') DEFAULT 'BEKLEMEDE',
                description TEXT,
                created_by INT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                INDEX idx_due_date (due_date),
                INDEX idx_status (status),
                INDEX idx_customer_name (customer_name),
                FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_turkish_ci
        ");
    }
}
