<?php
/**
 * BankKmhController - KMH (Kredili Mevduat Hesabı) Yönetimi
 *
 * KMH = Değişken bakiyeli kredi limiti (kredi kartı gibi)
 * Pattern: Mevcut BankController (limit + transactions)
 */

class BankKmhController {
    private $db;
    private $auth;

    public function __construct() {
        $this->db = Database::getInstance();
        $this->auth = new Auth();
    }

    /**
     * KMH hesap listesi
     */
    public function index() {
        $kmhAccounts = $this->db->select("
            SELECT
                b.id,
                b.bank_name,
                b.account_subtype,
                b.limit_amount,
                b.used_amount,
                b.available_amount,
                b.status,
                COALESCE(SUM(
                    CASE
                        WHEN bt.transaction_type = 'borrow' THEN bt.amount
                        WHEN bt.transaction_type = 'payment' THEN -bt.amount
                        ELSE 0
                    END
                ), 0) as total_balance,
                COUNT(bt.id) as transaction_count,
                MAX(bt.transaction_date) as last_transaction_date
            FROM banks b
            LEFT JOIN bank_transactions bt ON b.id = bt.bank_id
            WHERE b.account_type = 'kmh'
            GROUP BY b.id, b.bank_name, b.account_subtype, b.limit_amount, b.used_amount, b.available_amount, b.status
            ORDER BY b.bank_name ASC
        ");

        // Toplam özet
        $summary = $this->db->selectOne("
            SELECT
                COUNT(*) as account_count,
                SUM(limit_amount) as total_limit,
                SUM(used_amount) as total_used,
                SUM(available_amount) as total_available
            FROM banks
            WHERE account_type = 'kmh'
        ");

        $auth = $this->auth;
        $pageTitle = 'KMH Takibi';
        $currentPage = 'banks';

        require_once __DIR__ . '/../views/banks/kmh/index.php';
    }

    /**
     * KMH hesap detayı + işlem geçmişi
     */
    public function view($id) {
        $account = $this->db->selectOne("SELECT * FROM banks WHERE id = ? AND account_type = 'kmh'", [$id]);

        if (!$account) {
            Helper::setFlash('KMH hesabı bulunamadı!', 'danger');
            Helper::redirect('/banks/kmh');
        }

        $balance = $this->calculateBalance($id);

        $transactions = $this->db->select("
            SELECT * FROM bank_transactions
            WHERE bank_id = ?
            ORDER BY transaction_date DESC, created_at DESC
        ", [$id]);

        $auth = $this->auth;
        $pageTitle = $account['bank_name'];
        $currentPage = 'banks';

        require_once __DIR__ . '/../views/banks/kmh/view.php';
    }

    /**
     * Yeni KMH hesabı formu
     */
    public function create() {
        $auth = $this->auth;
        $pageTitle = 'Yeni KMH Hesabı';
        $currentPage = 'banks';

        require_once __DIR__ . '/../views/banks/kmh/create.php';
    }

    /**
     * KMH hesabı kaydetme
     */
    public function store() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/banks/kmh');
        }

        $data = [
            'bank_name' => $_POST['bank_name'] ?? '',
            'account_type' => 'kmh',
            'account_subtype' => $_POST['account_subtype'] ?? null,
            'limit_amount' => floatval($_POST['limit_amount'] ?? 0),
            'used_amount' => floatval($_POST['used_amount'] ?? 0),
            'status' => $_POST['status'] ?? 'BEKLEMEDE',
            'notes' => $_POST['notes'] ?? null
        ];

        if (empty($data['bank_name']) || $data['limit_amount'] <= 0) {
            Helper::setFlash('Banka adı ve limit tutarı zorunludur.', 'danger');
            Helper::redirect('/banks/kmh/create');
        }

        $id = $this->db->insert('banks', $data);

        if ($id) {
            // Eğer başlangıç borcu varsa, transaction oluştur
            if ($data['used_amount'] > 0) {
                $this->db->insert('bank_transactions', [
                    'bank_id' => $id,
                    'transaction_type' => 'borrow',
                    'amount' => $data['used_amount'],
                    'transaction_date' => date('Y-m-d'),
                    'description' => 'İlk borç kaydı',
                    'created_by' => $_SESSION['user_id'] ?? null
                ]);
            }

            Helper::setFlash('KMH hesabı başarıyla eklendi.', 'success');
            Helper::redirect('/banks/kmh');
        } else {
            Helper::setFlash('KMH hesabı eklenirken hata oluştu.', 'danger');
            Helper::redirect('/banks/kmh/create');
        }
    }

    /**
     * KMH hesabı düzenleme formu
     */
    public function edit($id) {
        $account = $this->db->selectOne("SELECT * FROM banks WHERE id = ? AND account_type = 'kmh'", [$id]);

        if (!$account) {
            Helper::setFlash('KMH hesabı bulunamadı!', 'danger');
            Helper::redirect('/banks/kmh');
        }

        $auth = $this->auth;
        $pageTitle = 'KMH Hesabı Düzenle';
        $currentPage = 'banks';

        require_once __DIR__ . '/../views/banks/kmh/edit.php';
    }

    /**
     * KMH hesabı güncelleme
     */
    public function update($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/banks/kmh');
        }

        $existing = $this->db->selectOne("SELECT * FROM banks WHERE id = ? AND account_type = 'kmh'", [$id]);
        if (!$existing) {
            Helper::setFlash('KMH hesabı bulunamadı!', 'danger');
            Helper::redirect('/banks/kmh');
        }

        $data = [
            'bank_name' => $_POST['bank_name'] ?? '',
            'account_subtype' => $_POST['account_subtype'] ?? null,
            'limit_amount' => floatval($_POST['limit_amount'] ?? 0),
            'used_amount' => floatval($_POST['used_amount'] ?? 0),
            'status' => $_POST['status'] ?? 'BEKLEMEDE',
            'notes' => $_POST['notes'] ?? null
        ];

        if (empty($data['bank_name']) || $data['limit_amount'] <= 0) {
            Helper::setFlash('Banka adı ve limit tutarı zorunludur.', 'danger');
            Helper::redirect('/banks/kmh/edit/' . $id);
        }

        $result = $this->db->update('banks', $data, 'id = ?', [$id]);

        if ($result !== false) {
            Helper::setFlash('KMH hesabı başarıyla güncellendi.', 'success');
            Helper::redirect('/banks/kmh');
        } else {
            Helper::setFlash('KMH hesabı güncellenirken hata oluştu.', 'danger');
            Helper::redirect('/banks/kmh/edit/' . $id);
        }
    }

    /**
     * Yeni işlem ekleme
     */
    public function saveTransaction($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/banks/kmh/view/' . $id);
        }

        $account = $this->db->selectOne("SELECT * FROM banks WHERE id = ? AND account_type = 'kmh'", [$id]);
        if (!$account) {
            Helper::setFlash('KMH hesabı bulunamadı!', 'danger');
            Helper::redirect('/banks/kmh');
        }

        $data = [
            'bank_id' => $id,
            'transaction_type' => $_POST['transaction_type'] ?? '',
            'amount' => floatval($_POST['amount'] ?? 0),
            'transaction_date' => $_POST['transaction_date'] ?? date('Y-m-d'),
            'reference_number' => $_POST['reference_number'] ?? null,
            'description' => $_POST['description'] ?? null,
            'created_by' => $_SESSION['user_id'] ?? null
        ];

        if (empty($data['transaction_type']) || $data['amount'] <= 0) {
            Helper::setFlash('İşlem tipi ve tutar gerekli!', 'danger');
            Helper::redirect('/banks/kmh/view/' . $id);
        }

        $result = $this->db->insert('bank_transactions', $data);

        if ($result) {
            // used_amount'u güncelle
            if ($data['transaction_type'] === 'borrow') {
                $this->db->query("UPDATE banks SET used_amount = used_amount + ? WHERE id = ?", [$data['amount'], $id]);
            } else if ($data['transaction_type'] === 'payment') {
                $this->db->query("UPDATE banks SET used_amount = used_amount - ? WHERE id = ?", [$data['amount'], $id]);
            }

            Helper::setFlash('İşlem başarıyla kaydedildi!', 'success');
        } else {
            Helper::setFlash('İşlem kaydedilemedi!', 'danger');
        }

        Helper::redirect('/banks/kmh/view/' . $id);
    }

    /**
     * İşlem silme
     */
    public function deleteTransaction($accountId, $transactionId) {
        $account = $this->db->selectOne("SELECT * FROM banks WHERE id = ? AND account_type = 'kmh'", [$accountId]);
        if (!$account) {
            Helper::setFlash('KMH hesabı bulunamadı!', 'danger');
            Helper::redirect('/banks/kmh');
        }

        $transaction = $this->db->selectOne("
            SELECT * FROM bank_transactions WHERE id = ? AND bank_id = ?
        ", [$transactionId, $accountId]);

        if (!$transaction) {
            Helper::setFlash('İşlem bulunamadı!', 'danger');
            Helper::redirect('/banks/kmh/view/' . $accountId);
        }

        // used_amount'u geri al
        if ($transaction['transaction_type'] === 'borrow') {
            $this->db->query("UPDATE banks SET used_amount = used_amount - ? WHERE id = ?", [$transaction['amount'], $accountId]);
        } else if ($transaction['transaction_type'] === 'payment') {
            $this->db->query("UPDATE banks SET used_amount = used_amount + ? WHERE id = ?", [$transaction['amount'], $accountId]);
        }

        $result = $this->db->delete('bank_transactions', 'id = ?', [$transactionId]);

        if ($result) {
            Helper::setFlash('İşlem başarıyla silindi!', 'success');
        } else {
            Helper::setFlash('İşlem silinemedi!', 'danger');
        }

        Helper::redirect('/banks/kmh/view/' . $accountId);
    }

    /**
     * KMH hesabı silme
     */
    public function delete($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/banks/kmh');
        }

        $result = $this->db->delete('banks', 'id = ? AND account_type = ?', [$id, 'kmh']);

        if ($result) {
            Helper::setFlash('KMH hesabı başarıyla silindi.', 'success');
        } else {
            Helper::setFlash('KMH hesabı silinemedi.', 'danger');
        }

        Helper::redirect('/banks/kmh');
    }

    /**
     * Bakiye hesaplama
     */
    private function calculateBalance($accountId) {
        $result = $this->db->selectOne("
            SELECT
                SUM(CASE
                    WHEN transaction_type = 'borrow' THEN amount
                    WHEN transaction_type = 'payment' THEN -amount
                    WHEN transaction_type = 'adjustment' THEN amount
                    ELSE 0
                END) as balance,
                SUM(CASE WHEN transaction_type = 'borrow' THEN amount ELSE 0 END) as total_borrows,
                SUM(CASE WHEN transaction_type = 'payment' THEN amount ELSE 0 END) as total_payments,
                COUNT(*) as transaction_count
            FROM bank_transactions
            WHERE bank_id = ?
        ", [$accountId]);

        return [
            'balance' => $result['balance'] ?? 0,
            'total_borrows' => $result['total_borrows'] ?? 0,
            'total_payments' => $result['total_payments'] ?? 0,
            'transaction_count' => $result['transaction_count'] ?? 0
        ];
    }
}
