<?php
/**
 * BankBchController - BCH (Banka Cari Hesabı) Yönetimi
 *
 * BCH = Tek seferlik sabit ödeme (senet gibi)
 * Pattern: SupplierController (tek borç takibi)
 */

class BankBchController {
    private $db;
    private $auth;

    public function __construct() {
        $this->db = Database::getInstance();
        $this->auth = new Auth();
    }

    /**
     * BCH listesi
     */
    public function index() {
        $bchPayments = $this->db->select("
            SELECT *
            FROM bank_bch
            ORDER BY due_date ASC, created_at DESC
        ");

        // Toplam hesapla
        $summary = $this->db->selectOne("
            SELECT
                COUNT(*) as total_count,
                SUM(CASE WHEN status = 'BEKLEMEDE' THEN amount ELSE 0 END) as pending_total,
                SUM(CASE WHEN status = 'ÖDEME' THEN amount ELSE 0 END) as paid_total
            FROM bank_bch
        ");

        $auth = $this->auth;
        $pageTitle = 'BCH Takibi';
        $currentPage = 'banks';

        require_once __DIR__ . '/../views/banks/bch/index.php';
    }

    /**
     * Yeni BCH formu
     */
    public function create() {
        $auth = $this->auth;
        $pageTitle = 'Yeni BCH Ekle';
        $currentPage = 'banks';

        require_once __DIR__ . '/../views/banks/bch/create.php';
    }

    /**
     * BCH düzenleme
     */
    public function edit($id) {
        $bch = $this->db->selectOne("SELECT * FROM bank_bch WHERE id = ?", [$id]);

        if (!$bch) {
            Helper::setFlash('BCH kaydı bulunamadı.', 'danger');
            Helper::redirect('/banks/bch');
        }

        $auth = $this->auth;
        $pageTitle = 'BCH Düzenle';
        $currentPage = 'banks';

        require_once __DIR__ . '/../views/banks/bch/edit.php';
    }

    /**
     * BCH kaydetme (POST handler for create)
     */
    public function store() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/banks/bch');
        }

        // Validate required fields
        $bank_name = $_POST['bank_name'] ?? '';
        $amount = $_POST['amount'] ?? 0;
        $due_date = $_POST['due_date'] ?? '';

        if (empty($bank_name) || empty($amount) || empty($due_date)) {
            Helper::setFlash('Banka adı, tutar ve vade tarihi zorunludur.', 'danger');
            Helper::redirect('/banks/bch/create');
        }

        // Prepare data
        $data = [
            'bank_name' => $bank_name,
            'amount' => floatval($amount),
            'due_date' => $due_date,
            'status' => $_POST['status'] ?? 'BEKLEMEDE',
            'payment_date' => !empty($_POST['payment_date']) ? $_POST['payment_date'] : null,
            'reference_number' => $_POST['reference_number'] ?? null,
            'description' => $_POST['description'] ?? null,
            'created_by' => $_SESSION['user_id'] ?? null
        ];

        // Insert to database
        $id = $this->db->insert('bank_bch', $data);

        if ($id) {
            Helper::setFlash('BCH kaydı başarıyla eklendi.', 'success');
            Helper::redirect('/banks/bch');
        } else {
            Helper::setFlash('BCH kaydı eklenirken hata oluştu.', 'danger');
            Helper::redirect('/banks/bch/create');
        }
    }

    /**
     * BCH güncelleme (POST handler for edit)
     */
    public function update($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/banks/bch');
        }

        // Check if BCH exists
        $existing = $this->db->selectOne("SELECT * FROM bank_bch WHERE id = ?", [$id]);
        if (!$existing) {
            Helper::setFlash('BCH kaydı bulunamadı.', 'danger');
            Helper::redirect('/banks/bch');
        }

        // Validate required fields
        $bank_name = $_POST['bank_name'] ?? '';
        $amount = $_POST['amount'] ?? 0;
        $due_date = $_POST['due_date'] ?? '';

        if (empty($bank_name) || empty($amount) || empty($due_date)) {
            Helper::setFlash('Banka adı, tutar ve vade tarihi zorunludur.', 'danger');
            Helper::redirect('/banks/bch/edit/' . $id);
        }

        // Prepare data
        $data = [
            'bank_name' => $bank_name,
            'amount' => floatval($amount),
            'due_date' => $due_date,
            'status' => $_POST['status'] ?? 'BEKLEMEDE',
            'payment_date' => !empty($_POST['payment_date']) ? $_POST['payment_date'] : null,
            'reference_number' => $_POST['reference_number'] ?? null,
            'description' => $_POST['description'] ?? null,
        ];

        // Update database
        $result = $this->db->update('bank_bch', $data, 'id = ?', [$id]);

        if ($result !== false) {
            Helper::setFlash('BCH kaydı başarıyla güncellendi.', 'success');
            Helper::redirect('/banks/bch');
        } else {
            Helper::setFlash('BCH kaydı güncellenirken hata oluştu.', 'danger');
            Helper::redirect('/banks/bch/edit/' . $id);
        }
    }

    /**
     * BCH silme
     */
    public function delete($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/banks/bch');
        }

        $result = $this->db->delete('bank_bch', 'id = ?', [$id]);

        if ($result) {
            Helper::setFlash('BCH kaydı başarıyla silindi.', 'success');
        } else {
            Helper::setFlash('BCH kaydı silinemedi.', 'danger');
        }

        Helper::redirect('/banks/bch');
    }

    /**
     * BCH'yi ödendi olarak işaretle
     */
    public function markAsPaid($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/banks/bch');
        }

        $bch = $this->db->selectOne("SELECT * FROM bank_bch WHERE id = ?", [$id]);
        if (!$bch) {
            Helper::setFlash('BCH kaydı bulunamadı.', 'danger');
            Helper::redirect('/banks/bch');
        }

        $data = [
            'status' => 'ÖDEME',
            'payment_date' => date('Y-m-d')
        ];

        $result = $this->db->update('bank_bch', $data, 'id = ?', [$id]);

        if ($result !== false) {
            Helper::setFlash('BCH ödendi olarak işaretlendi.', 'success');
        } else {
            Helper::setFlash('İşlem başarısız.', 'danger');
        }

        Helper::redirect('/banks/bch');
    }
}
